#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os
import sys
import json
import subprocess
from datetime import datetime

print("Content-Type: application/json\n")

# --- Récupération des paramètres GET ---
query = os.environ.get("QUERY_STRING", "")
params = dict(p.split("=", 1) for p in query.split("&") if "=" in p)

session_id = params.get("sid")
asset_id = params.get("aid")

if not session_id:
    print(json.dumps({"error": "missing sid"}))
    sys.exit()

# --- Construction du chemin absolu vers analyze.py ---
SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
ANALYZE_PATH = os.path.join(SCRIPT_DIR, "analyze.py")

if not os.path.exists(ANALYZE_PATH):
    print(json.dumps({"error": f"analyze.py not found at {ANALYZE_PATH}"}))
    sys.exit()

# --- Appel de analyze.py via CGI ---
try:
    env = os.environ.copy()
    env["QUERY_STRING"] = f"sid={session_id}"
    if asset_id:
        env["QUERY_STRING"] += f"&aid={asset_id}"

    result = subprocess.run(
        ["python3", ANALYZE_PATH],
        env=env,
        capture_output=True,
        text=True
    )

    if result.returncode != 0:
        raise subprocess.CalledProcessError(result.returncode, result.args, result.stdout, result.stderr)

    raw_output = result.stdout.strip()

except subprocess.CalledProcessError as e:
    print(json.dumps({
        "error": f"analyze.py call failed: {str(e)}",
        "stdout": e.stdout,
        "stderr": e.stderr
    }))
    sys.exit()

# --- ⚙️ Supprimer les en-têtes CGI avant parsing ---
json_start = raw_output.find("{")
if json_start == -1:
    print(json.dumps({
        "error": "invalid output from analyze.py (no JSON found)",
        "stdout": raw_output
    }))
    sys.exit()

clean_json = raw_output[json_start:]

try:
    data = json.loads(clean_json)
except Exception as e:
    print(json.dumps({"error": f"Failed to parse analyze.py output: {str(e)}", "stdout": clean_json}))
    sys.exit()

assets = data.get("assets", {})
transactions = data.get("transactions", {})

def convert_period_to_date(period_str):
    """Convertit YYYYMM en YYYY-MM-01"""
    try:
        return datetime.strptime(period_str, "%Y%m").strftime("%Y-%m-01")
    except Exception:
        return None

graph_data = {}

for aid, asset in assets.items():
    if asset_id and aid != asset_id:
        continue

    history = asset.get("history", {})
    txs = transactions.get(aid, [])

    # Trouver la première transaction d'achat
    first_buy = None
    for t in txs:
        if "Achat" in t.get("type", ""):
            first_buy = t
            break

    # Construire la chronologie
    timeline = []

    # Ajouter la première transaction d'achat (si existante)
    if first_buy:
        timeline.append({
            "date": first_buy["date_iso"],
            "price": first_buy.get("price_per_splint"),
            "source": "transaction"
        })

    # Ajouter les historiques d’évolution du prix
    for period, values in history.items():
        dt = convert_period_to_date(period)
        if dt:
            timeline.append({
                "date": dt,
                "price": values.get("current_splint_value", None),
                "source": "history"
            })

    # Trier par date croissante
    timeline.sort(key=lambda x: x["date"])

    graph_data[aid] = {
        "name": asset.get("name"),
        "timeline": timeline
    }

# --- Sortie JSON finale ---
print(json.dumps({
    "session_id": session_id,
    "assets": graph_data
}, ensure_ascii=False, indent=2))
